﻿using System;
using System.IO;
using CalamityMod.Buffs.DamageOverTime;
using CalamityMod.Dusts;
using CalamityMod.Events;
using CalamityMod.Items.Accessories;
using CalamityMod.Items.Weapons.Typeless;
using CalamityMod.Projectiles.Boss;
using CalamityMod.World;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using ReLogic.Content;
using Terraria;
using Terraria.Audio;
using Terraria.GameContent;
using Terraria.GameContent.Bestiary;
using Terraria.ID;
using Terraria.ModLoader;
using Filters = Terraria.Graphics.Effects.Filters;

namespace CalamityMod.NPCs.ProfanedGuardians
{
    [AutoloadBossHead]
    public class ProfanedGuardianDefender : ModNPC
    {
        private int healTimer = 0;
        private const float TimeForShieldDespawn = 120f;
        public static readonly SoundStyle DashSound = new("CalamityMod/Sounds/Custom/ProfanedGuardians/GuardianDash");
        public static readonly SoundStyle RockShieldSpawnSound = new("CalamityMod/Sounds/Custom/ProfanedGuardians/GuardianRockShieldActivate");
        public static readonly SoundStyle ShieldDeathSound = new("CalamityMod/Sounds/Custom/ProfanedGuardians/GuardianShieldDeactivate");

        public static Asset<Texture2D> Texture_Glow;
        public static Asset<Texture2D> TextureNight_Glow;

        public override void SetStaticDefaults()
        {
            Main.npcFrameCount[NPC.type] = 10;
            NPCID.Sets.TrailingMode[NPC.type] = 1;
            NPCID.Sets.BossBestiaryPriority.Add(Type);
            NPCID.Sets.NPCBestiaryDrawModifiers value = new NPCID.Sets.NPCBestiaryDrawModifiers()
            {
                PortraitPositionXOverride = 0,
                PortraitScale = 0.75f,
                Scale = 0.75f
            };
            value.Position.X += 25;
            value.Position.Y += 15;
            NPCID.Sets.NPCBestiaryDrawOffset[Type] = value;
            if (!Main.dedServ)
            {
                Texture_Glow = ModContent.Request<Texture2D>(Texture + "Glow", AssetRequestMode.AsyncLoad);
                TextureNight_Glow = ModContent.Request<Texture2D>(Texture + "GlowNight", AssetRequestMode.AsyncLoad);
            }
        }

        public override void SetDefaults()
        {
            NPC.BossBar = Main.BigBossProgressBar.NeverValid;
            NPC.Calamity().canBreakPlayerDefense = true;
            NPC.npcSlots = 3f;
            NPC.aiStyle = -1;
            NPC.GetNPCDamage();
            NPC.width = 228;
            NPC.height = 164;
            NPC.defense = 50;
            NPC.DR_NERD(0.4f);
            NPC.LifeMaxNERB(40000, 48000, 35000);
            double HPBoost = CalamityServerConfig.Instance.BossHealthBoost * 0.01;
            NPC.lifeMax += (int)(NPC.lifeMax * HPBoost);
            NPC.knockBackResist = 0f;
            NPC.noGravity = true;
            NPC.noTileCollide = true;
            AIType = -1;
            NPC.HitSound = SoundID.NPCHit52;
            NPC.DeathSound = SoundID.NPCDeath55;
            NPC.Calamity().VulnerableToHeat = false;
            NPC.Calamity().VulnerableToCold = true;
            NPC.Calamity().VulnerableToSickness = false;
            NPC.Calamity().VulnerableToWater = true;

            // Scale stats in Expert and Master
            CalamityGlobalNPC.AdjustExpertModeStatScaling(NPC);
            CalamityGlobalNPC.AdjustMasterModeStatScaling(NPC);
        }

        public override void SetBestiary(BestiaryDatabase database, BestiaryEntry bestiaryEntry)
        {
            int associatedNPCType = ModContent.NPCType<ProfanedGuardianCommander>();
            bestiaryEntry.UIInfoProvider = new CommonEnemyUICollectionInfoProvider(ContentSamples.NpcBestiaryCreditIdsByNpcNetIds[associatedNPCType], quickUnlock: true);

            bestiaryEntry.Info.AddRange(new IBestiaryInfoElement[]
            {
                BestiaryDatabaseNPCsPopulator.CommonTags.SpawnConditions.Biomes.TheHallow,
                BestiaryDatabaseNPCsPopulator.CommonTags.SpawnConditions.Biomes.TheUnderworld,
                new FlavorTextBestiaryInfoElement("Mods.CalamityMod.Bestiary.ProfanedGuardianDefender")
            });
        }

        public override void SendExtraAI(BinaryWriter writer)
        {
            writer.Write(healTimer);
            writer.Write(NPC.chaseable);
            writer.Write(NPC.localAI[0]);
            writer.Write(NPC.localAI[1]);
            writer.Write(NPC.localAI[2]);
            writer.Write(NPC.localAI[3]);
        }

        public override void ReceiveExtraAI(BinaryReader reader)
        {
            healTimer = reader.ReadInt32();
            NPC.chaseable = reader.ReadBoolean();
            NPC.localAI[0] = reader.ReadSingle();
            NPC.localAI[1] = reader.ReadSingle();
            NPC.localAI[2] = reader.ReadSingle();
            NPC.localAI[3] = reader.ReadSingle();
        }

        public override void FindFrame(int frameHeight)
        {
            NPC.frameCounter += 0.12f + NPC.velocity.Length() / 120f;
            NPC.frameCounter %= Main.npcFrameCount[NPC.type];
            int frame = (int)NPC.frameCounter;
            NPC.frame.Y = frame * frameHeight;
        }

        public override void AI()
        {
            CalamityGlobalNPC.doughnutBossDefender = NPC.whoAmI;

            Lighting.AddLight((int)((NPC.position.X + (NPC.width / 2)) / 16f), (int)((NPC.position.Y + (NPC.height / 2)) / 16f), 1.1f, 0.9f, 0f);

            if (CalamityGlobalNPC.doughnutBoss < 0 || !Main.npc[CalamityGlobalNPC.doughnutBoss].active)
            {
                NPC.life = 0;
                NPC.HitEffect();
                NPC.active = false;
                NPC.netUpdate = true;
                return;
            }

            // Projectile and dust spawn location variables
            Vector2 dustAndProjectileOffset = new Vector2(40f * NPC.direction, 20f);
            Vector2 shootFrom = NPC.Center + dustAndProjectileOffset;

            // Rotation
            NPC.rotation = NPC.velocity.X * 0.005f;

            bool healerAlive = false;
            if (CalamityGlobalNPC.doughnutBossHealer != -1)
            {
                if (Main.npc[CalamityGlobalNPC.doughnutBossHealer].active)
                    healerAlive = true;
            }

            // Healing
            if (healerAlive)
            {
                float distanceFromHealer = Vector2.Distance(Main.npc[CalamityGlobalNPC.doughnutBossHealer].Center, NPC.Center);
                bool dontHeal = distanceFromHealer > 2000f || Main.npc[CalamityGlobalNPC.doughnutBossHealer].justHit || NPC.life == NPC.lifeMax;
                if (dontHeal)
                {
                    healTimer = 0;
                }
                else
                {
                    float healGateValue = 60f;
                    healTimer++;
                    if (healTimer >= healGateValue)
                    {
                        SoundEngine.PlaySound(SoundID.Item8, shootFrom);

                        int maxHealDustIterations = (int)distanceFromHealer;
                        int maxDust = 100;
                        int dustDivisor = maxHealDustIterations / maxDust;
                        if (dustDivisor < 2)
                            dustDivisor = 2;

                        Vector2 healDustOffset = new Vector2(40f * Main.npc[CalamityGlobalNPC.doughnutBossHealer].direction, 20f);
                        Vector2 dustLineStart = Main.npc[CalamityGlobalNPC.doughnutBossHealer].Center + healDustOffset;
                        Vector2 dustLineEnd = shootFrom;
                        Vector2 currentDustPos = default;
                        Vector2 spinningpoint = new Vector2(0f, -3f).RotatedByRandom(MathHelper.Pi);
                        Vector2 value5 = new Vector2(2.1f, 2f);
                        int dustSpawned = 0;
                        for (int i = 0; i < maxHealDustIterations; i++)
                        {
                            if (i % dustDivisor == 0)
                            {
                                currentDustPos = Vector2.Lerp(dustLineStart, dustLineEnd, i / (float)maxHealDustIterations);
                                Color dustColor = Main.hslToRgb(Main.rgbToHsl(new Color(255, 200, Math.Abs(Main.DiscoB - (int)(dustSpawned * 2.55f)))).X, 1f, 0.5f);
                                dustColor.A = 255;
                                int dust = Dust.NewDust(currentDustPos, 0, 0, DustID.RainbowMk2, 0f, 0f, 0, dustColor, 1f);
                                Main.dust[dust].position = currentDustPos;
                                Main.dust[dust].velocity = spinningpoint.RotatedBy(MathHelper.TwoPi * i / maxHealDustIterations) * value5 * (0.8f + Main.rand.NextFloat() * 0.4f) + NPC.velocity;
                                Main.dust[dust].noGravity = true;
                                Main.dust[dust].scale = 1f;
                                Main.dust[dust].fadeIn = Main.rand.NextFloat() * 2f;
                                Dust dust2 = Dust.CloneDust(dust);
                                Dust dust3 = dust2;
                                dust3.scale /= 2f;
                                dust3 = dust2;
                                dust3.fadeIn /= 2f;
                                dust2.color = new Color(255, 255, 255, 255);
                                dustSpawned++;
                            }
                        }

                        healTimer = 0;
                        if (Main.netMode != NetmodeID.MultiplayerClient)
                        {
                            int healAmt = NPC.lifeMax / 10;
                            if (healAmt > NPC.lifeMax - NPC.life)
                                healAmt = NPC.lifeMax - NPC.life;

                            if (healAmt > 0)
                            {
                                NPC.life += healAmt;
                                NPC.HealEffect(healAmt, true);
                                NPC.netUpdate = true;
                            }
                        }
                    }
                }

                if (Main.npc[CalamityGlobalNPC.doughnutBossHealer].ai[0] == 599 && Main.zenithWorld && Main.netMode != NetmodeID.MultiplayerClient)
                {
                    // gain more health once the healer's channel heal is done
                    NPC.lifeMax += 7500;
                    NPC.life += NPC.lifeMax - NPC.life;
                    NPC.HealEffect(NPC.lifeMax - NPC.life, true);
                    NPC.netUpdate = true;
                }
            }

            // Despawn
            if (Main.npc[CalamityGlobalNPC.doughnutBoss].ai[3] == -1f)
            {
                NPC.velocity = Main.npc[CalamityGlobalNPC.doughnutBoss].velocity;
                return;
            }

            // Get the Guardian Commander's target
            Player player = Main.player[Main.npc[CalamityGlobalNPC.doughnutBoss].target];

            bool bossRush = BossRushEvent.BossRushActive;
            bool expertMode = Main.expertMode || bossRush;
            bool revenge = CalamityWorld.revenge || bossRush;
            bool death = CalamityWorld.death || bossRush;

            // Enrage at nighttime
            bool enraged = false;
            if (!Main.dayTime && !Main.remixWorld)
            {
                enraged = true;
                NPC.Calamity().CurrentlyEnraged = true;
            }

            bool phase1 = healerAlive;

            NPC.chaseable = !phase1;

            // Phase durations
            float commanderGuardPhase2Duration = (bossRush || enraged) ? 420f : death ? 480f : revenge ? 510f : expertMode ? 540f : 600f;
            float timeBeforeRocksRespawnInPhase2 = 90f;
            float throwRocksGateValue = 60f;

            // Distance
            float distanceInFrontOfCommander = 160f;

            // Charge variables
            float chargeVelocityMult = 0.25f;
            float maxChargeVelocity = (bossRush || enraged) ? 25f : death ? 22f : revenge ? 20.5f : expertMode ? 19f : 16f;
            if (Main.getGoodWorld)
                maxChargeVelocity *= 1.15f;

            // Whether the commander is calling all guardians together for the laser attack
            bool commanderUsingLaser = Main.npc[CalamityGlobalNPC.doughnutBoss].ai[0] == 5f;

            // Go low just before moving to the other side to avoid bullshit hits
            float moveToOtherSideInPhase1GateValue = 900f;
            float timeBeforeMoveToOtherSideInPhase1Reset = moveToOtherSideInPhase1GateValue * 2f;
            float totalGoLowDurationPhase1 = 240f;
            float goLowDurationPhase1 = totalGoLowDurationPhase1 * 0.5f;
            float roundedGoLowPhase1Check = (float)Math.Round(goLowDurationPhase1 * 0.5);
            bool commanderGoingLowOrHighInPhase1 = (Main.npc[CalamityGlobalNPC.doughnutBoss].localAI[3] > (moveToOtherSideInPhase1GateValue - goLowDurationPhase1) &&
                                                    Main.npc[CalamityGlobalNPC.doughnutBoss].localAI[3] <= (moveToOtherSideInPhase1GateValue + roundedGoLowPhase1Check)) ||
                                                   Main.npc[CalamityGlobalNPC.doughnutBoss].localAI[3] > (timeBeforeMoveToOtherSideInPhase1Reset - goLowDurationPhase1) ||
                                                   Main.npc[CalamityGlobalNPC.doughnutBoss].localAI[3] <= (-roundedGoLowPhase1Check);

            float moveToOtherSideInPhase2GateValue = commanderGuardPhase2Duration - 120f;
            float timeBeforeMoveToOtherSideInPhase2Reset = moveToOtherSideInPhase2GateValue * 2f;
            float totalGoLowDurationPhase2 = 210f;
            float goLowDurationPhase2 = totalGoLowDurationPhase2 * 0.5f;
            float roundedGoLowPhase2Check = (float)Math.Round(goLowDurationPhase2 * 0.5);
            bool commanderGoingLowOrHighInPhase2 = (Main.npc[CalamityGlobalNPC.doughnutBoss].Calamity().newAI[1] > (moveToOtherSideInPhase2GateValue - goLowDurationPhase2) &&
                                                    Main.npc[CalamityGlobalNPC.doughnutBoss].Calamity().newAI[1] <= (moveToOtherSideInPhase2GateValue + roundedGoLowPhase2Check)) ||
                                                   Main.npc[CalamityGlobalNPC.doughnutBoss].Calamity().newAI[1] > (timeBeforeMoveToOtherSideInPhase2Reset - goLowDurationPhase2) ||
                                                   Main.npc[CalamityGlobalNPC.doughnutBoss].Calamity().newAI[1] <= (-roundedGoLowPhase2Check);

            // Tell rocks to fade out and shrink
            if (commanderGoingLowOrHighInPhase1 || commanderGoingLowOrHighInPhase2)
                NPC.localAI[3] = 1f;
            else
                NPC.localAI[3] = 0f;

            // Spawn rock shield
            bool respawnRocksInPhase2 = NPC.ai[1] == (-commanderGuardPhase2Duration + timeBeforeRocksRespawnInPhase2) && !commanderGoingLowOrHighInPhase2;
            int rockTypes = 6;
            int maxRocks = respawnRocksInPhase2 ? 18 : 36;
            int rockRings = 3;
            int totalRocksPerRing = maxRocks / rockRings;
            int spacing = 360 / totalRocksPerRing;
            int distance2 = 200;
            bool justSpawnedRocks = false;
            if (NPC.localAI[0] == 0f || respawnRocksInPhase2)
            {
                justSpawnedRocks = true;
                NPC.localAI[0] = 1f;
                if (Main.netMode != NetmodeID.MultiplayerClient)
                {
                    for (int i = 0; i < totalRocksPerRing; i++)
                    {
                        int rockType = Main.rand.Next(rockTypes) + 1;
                        NPC.NewNPC(NPC.GetSource_FromAI(), (int)(NPC.Center.X + (Math.Sin(i * spacing) * distance2)), (int)(NPC.Center.Y + (Math.Cos(i * spacing) * distance2)), ModContent.NPCType<ProfanedRocks>(), NPC.whoAmI, i * spacing, 0f, rockType, 0f);
                        rockType = Main.rand.Next(rockTypes) + 1;
                        NPC.NewNPC(NPC.GetSource_FromAI(), (int)(NPC.Center.X + (Math.Sin(i * spacing) * distance2)), (int)(NPC.Center.Y + (Math.Cos(i * spacing) * distance2)), ModContent.NPCType<ProfanedRocks>(), NPC.whoAmI, i * spacing, 1f, rockType, 0f);
                        rockType = Main.rand.Next(rockTypes) + 1;
                        NPC.NewNPC(NPC.GetSource_FromAI(), (int)(NPC.Center.X + (Math.Sin(i * spacing) * distance2)), (int)(NPC.Center.Y + (Math.Cos(i * spacing) * distance2)), ModContent.NPCType<ProfanedRocks>(), NPC.whoAmI, i * spacing, 2f, rockType, 0f);
                    }
                }
            }

            // Generate new rock shields if too many are broken in phase 1
            if (phase1)
            {
                int minRocks = maxRocks / 2;
                int numRockShields = NPC.CountNPCS(ModContent.NPCType<ProfanedRocks>());
                if (numRockShields < minRocks)
                {
                    justSpawnedRocks = true;
                    totalRocksPerRing = minRocks / rockRings;
                    spacing = 360 / totalRocksPerRing;
                    if (Main.netMode != NetmodeID.MultiplayerClient)
                    {
                        for (int i = 0; i < totalRocksPerRing; i++)
                        {
                            int rockType = Main.rand.Next(rockTypes) + 1;
                            NPC.NewNPC(NPC.GetSource_FromAI(), (int)(NPC.Center.X + (Math.Sin(i * spacing) * distance2)), (int)(NPC.Center.Y + (Math.Cos(i * spacing) * distance2)), ModContent.NPCType<ProfanedRocks>(), NPC.whoAmI, i * spacing, 0f, rockType, 0f);
                            rockType = Main.rand.Next(rockTypes) + 1;
                            NPC.NewNPC(NPC.GetSource_FromAI(), (int)(NPC.Center.X + (Math.Sin(i * spacing) * distance2)), (int)(NPC.Center.Y + (Math.Cos(i * spacing) * distance2)), ModContent.NPCType<ProfanedRocks>(), NPC.whoAmI, i * spacing, 1f, rockType, 0f);
                            rockType = Main.rand.Next(rockTypes) + 1;
                            NPC.NewNPC(NPC.GetSource_FromAI(), (int)(NPC.Center.X + (Math.Sin(i * spacing) * distance2)), (int)(NPC.Center.Y + (Math.Cos(i * spacing) * distance2)), ModContent.NPCType<ProfanedRocks>(), NPC.whoAmI, i * spacing, 2f, rockType, 0f);
                        }
                    }
                }
            }
            else
            {
                if (NPC.localAI[1] < TimeForShieldDespawn)
                {
                    // Star Wrath use sound
                    if (NPC.localAI[1] == 0f)
                        SoundEngine.PlaySound(ShieldDeathSound, NPC.Center);

                    NPC.localAI[1] += 1f;
                }
            }

            // Spawn three dust circles and play noise whenever rock shields are spawned
            if (justSpawnedRocks)
            {
                // Meteor Staff use sound and dust circles
                SoundEngine.PlaySound(RockShieldSpawnSound, NPC.Center);
                int totalDust = maxRocks;
                for (int j = 0; j < rockRings; j++)
                {
                    for (int k = 0; k < totalDust; k++)
                    {
                        Vector2 dustSpawnPos = NPC.velocity.SafeNormalize(Vector2.UnitY) * new Vector2(distance2, distance2);
                        dustSpawnPos = dustSpawnPos.RotatedBy((double)((k - (totalDust / 2 - 1)) * MathHelper.TwoPi / totalDust), default) + NPC.Center;
                        Vector2 dustVelocity = dustSpawnPos - NPC.Center;
                        Color dustColor = Main.hslToRgb(Main.rgbToHsl(Color.Orange).X, 1f, 0.5f);
                        dustColor.A = 255;
                        int dust = Dust.NewDust(dustSpawnPos + dustVelocity, 0, 0, DustID.RainbowMk2, dustVelocity.X, dustVelocity.Y, 0, dustColor, 1.4f);
                        Main.dust[dust].noGravity = true;
                        Main.dust[dust].noLight = true;
                        Main.dust[dust].velocity = dustVelocity * (j * 0.1f + 0.1f);
                    }
                }
            }

            float moveVelocity = (bossRush || enraged) ? 24f : death ? 22f : revenge ? 21f : expertMode ? 20f : 18f;
            if (Main.getGoodWorld)
                moveVelocity *= 1.25f;
            if (healerAlive)
                moveVelocity *= 0.8f;

            float distanceToStayAwayFromTarget = 800f;
            bool speedUp = Vector2.Distance(NPC.Center, player.Center) > (distanceToStayAwayFromTarget + 160f);
            if (speedUp)
                moveVelocity *= 2f;
            if (commanderGoingLowOrHighInPhase2)
                moveVelocity *= 2f;
            else if (commanderGoingLowOrHighInPhase1)
                moveVelocity *= 1.66f;

            if (NPC.ai[0] == 0f)
            {
                // Avoid cheap bullshit
                NPC.damage = 0;

                // Face the target
                if (Math.Abs(NPC.Center.X - player.Center.X) > 10f)
                {
                    float playerLocation = NPC.Center.X - player.Center.X;
                    NPC.direction = playerLocation < 0f ? 1 : -1;
                    NPC.spriteDirection = NPC.direction;
                }

                // Slow down and tell the Profaned Rocks to spin and fling themselves at the target by setting NPC.ai[0] = 1f
                if (!phase1)
                {
                    NPC.velocity *= 0.9f;
                    if (NPC.velocity.Length() <= 2f)
                        NPC.velocity = Vector2.Zero;

                    NPC.ai[3] += 1f;
                    if (NPC.ai[3] >= throwRocksGateValue)
                    {
                        NPC.ai[0] = 1f;
                        NPC.ai[1] = 0f;
                        NPC.ai[3] = 0f;
                        NPC.netUpdate = true;
                    }

                    return;
                }

                // Lay a holy bomb every once in a while in phase 1 and while not doing the laser attack
                if (!commanderUsingLaser)
                {
                    float projectileShootGateValue = (bossRush || enraged) ? 420f : death ? 480f : revenge ? 510f : expertMode ? 540f : 600f;
                    NPC.ai[1] += 1f;
                    if (NPC.ai[1] >= projectileShootGateValue)
                    {
                        NPC.ai[1] = 0f;
                        if (Main.netMode != NetmodeID.MultiplayerClient)
                        {
                            float projectileVelocityY = NPC.velocity.Y;
                            if (projectileVelocityY < 0f)
                                projectileVelocityY = 0f;

                            projectileVelocityY += expertMode ? 4f : 3f;
                            Vector2 projectileVelocity = new Vector2(NPC.velocity.X * 0.25f, projectileVelocityY);
                            int type = ModContent.ProjectileType<HolyBomb>();
                            int damage = NPC.GetProjectileDamage(type);
                            Projectile.NewProjectile(NPC.GetSource_FromAI(), shootFrom, projectileVelocity, type, damage, 0f, Main.myPlayer);
                        }
                    }
                }

                // Defend the commander
                Vector2 distanceFromDestination = Main.npc[CalamityGlobalNPC.doughnutBoss].Center + Vector2.UnitX * ((commanderUsingLaser || commanderGoingLowOrHighInPhase1) ? 0f : distanceInFrontOfCommander) * Main.npc[CalamityGlobalNPC.doughnutBoss].direction - NPC.Center;
                Vector2 desiredVelocity = distanceFromDestination.SafeNormalize(new Vector2(NPC.direction, 0f)) * moveVelocity;
                if (distanceFromDestination.Length() > 40f)
                {
                    float inertia = (commanderUsingLaser || commanderGoingLowOrHighInPhase1) ? 10f : 15f;
                    if (Main.getGoodWorld)
                        inertia *= 0.8f;

                    NPC.velocity = (NPC.velocity * (inertia - 1) + desiredVelocity) / inertia;
                }
                else
                    NPC.velocity *= 0.9f;
            }

            // Phase 2
            // Throw rocks, charge, summon holy bombs and shoot molten blasts
            else if (NPC.ai[0] == 1f)
            {
                // Avoid cheap bullshit
                NPC.damage = 0;

                // Face the target
                if (Math.Abs(NPC.Center.X - player.Center.X) > 10f)
                {
                    float playerLocation = NPC.Center.X - player.Center.X;
                    NPC.direction = playerLocation < 0f ? 1 : -1;
                    NPC.spriteDirection = NPC.direction;
                }

                // Do not increment the phase timer while swapping sides along with the commander in phase 2
                if (!commanderGoingLowOrHighInPhase2)
                    NPC.ai[1] += 1f;

                // Slow down before throwing rock shields
                if (NPC.ai[1] >= -throwRocksGateValue)
                {
                    NPC.velocity *= 0.8f;
                    if (Main.getGoodWorld)
                        NPC.velocity *= 0.5f;
                }

                // Defend the commander
                else
                {
                    // Shoot molten blasts
                    int moltenBlastsDivisor = 4;
                    float shootMoltenBlastsGateValue = commanderGuardPhase2Duration / moltenBlastsDivisor;
                    if (NPC.ai[1] % shootMoltenBlastsGateValue == 0f && !commanderGoingLowOrHighInPhase2)
                    {
                        float moltenBlastVelocity = (bossRush || enraged) ? 18f : death ? 16f : revenge ? 15f : expertMode ? 14f : 12f;
                        int projTimeLeft = (int)(2400f / moltenBlastVelocity);
                        Vector2 velocity = Vector2.Normalize(player.Center - shootFrom) * moltenBlastVelocity;
                        if (Main.netMode != NetmodeID.MultiplayerClient)
                        {
                            int type = ModContent.ProjectileType<MoltenBlast>();
                            int damage = NPC.GetProjectileDamage(type);
                            if (Main.netMode != NetmodeID.MultiplayerClient)
                            {
                                int proj = Projectile.NewProjectile(NPC.GetSource_FromAI(), shootFrom, velocity, type, damage, 0f, Main.myPlayer, player.position.X, player.position.Y);
                                Main.projectile[proj].timeLeft = projTimeLeft;
                            }
                        }

                        // Dust for blasting out the molten blasts
                        for (int i = 0; i < 50; i++)
                        {
                            int dustID;
                            switch (Main.rand.Next(6))
                            {
                                case 0:
                                case 1:
                                case 2:
                                case 3:
                                    dustID = (int)CalamityDusts.ProfanedFire;
                                    break;
                                default:
                                    dustID = DustID.OrangeTorch;
                                    break;
                            }

                            // Choose a random speed and angle to blast the fire out
                            float dustSpeed = Main.rand.NextFloat(moltenBlastVelocity * 0.5f, moltenBlastVelocity);
                            float angleRandom = 0.06f;
                            Vector2 dustVel = new Vector2(dustSpeed, 0.0f).RotatedBy(velocity.ToRotation());
                            dustVel = dustVel.RotatedBy(-angleRandom);
                            dustVel = dustVel.RotatedByRandom(2.0f * angleRandom);

                            // Pick a size for the fire particles
                            float scale = Main.rand.NextFloat(1f, 2f);

                            // Actually spawn the fire
                            int idx = Dust.NewDust(shootFrom, 42, 42, dustID, dustVel.X, dustVel.Y, 0, default, scale);
                            Main.dust[idx].noGravity = true;
                        }

                        NPC.velocity = -velocity * 0.5f;
                    }

                    Vector2 distanceFromDestination = Main.npc[CalamityGlobalNPC.doughnutBoss].Center + (commanderGoingLowOrHighInPhase2 ? Vector2.Zero : (Vector2.UnitX * distanceInFrontOfCommander * Main.npc[CalamityGlobalNPC.doughnutBoss].direction)) - NPC.Center;
                    Vector2 desiredVelocity = distanceFromDestination.SafeNormalize(new Vector2(NPC.direction, 0f)) * moveVelocity;
                    if (distanceFromDestination.Length() > 40f)
                    {
                        float inertia = commanderGoingLowOrHighInPhase2 ? 8f : 15f;
                        if (Main.getGoodWorld)
                            inertia *= 0.8f;

                        NPC.velocity = (NPC.velocity * (inertia - 1) + desiredVelocity) / inertia;
                    }
                    else
                        NPC.velocity *= 0.9f;
                }

                // Charge at target
                if (NPC.ai[1] >= 0f)
                {
                    // Set damage
                    NPC.damage = NPC.defDamage;

                    NPC.ai[0] = 2f;
                    NPC.ai[1] = 0f;
                    NPC.netUpdate = true;
                    Vector2 targetVector = player.Center - NPC.Center;
                    Vector2 velocity = targetVector.SafeNormalize(new Vector2(NPC.direction, 0f));
                    velocity *= maxChargeVelocity;

                    // Start slow and accelerate
                    NPC.velocity = velocity * chargeVelocityMult;

                    // Dust ring and sound right as charge begins
                    SoundEngine.PlaySound(DashSound, NPC.Center);
                    int totalDust = 36;
                    for (int k = 0; k < totalDust; k++)
                    {
                        Vector2 dustSpawnPos = NPC.velocity.SafeNormalize(Vector2.UnitY) * new Vector2(160f, 160f);
                        dustSpawnPos = dustSpawnPos.RotatedBy((double)((k - (totalDust / 2 - 1)) * MathHelper.TwoPi / totalDust), default) + shootFrom;
                        Vector2 dustVelocity = dustSpawnPos - shootFrom;
                        int dust = Dust.NewDust(dustSpawnPos + dustVelocity, 0, 0, (int)CalamityDusts.ProfanedFire, dustVelocity.X, dustVelocity.Y, 0, default, 1f);
                        Main.dust[dust].noGravity = true;
                        Main.dust[dust].noLight = true;
                        Main.dust[dust].scale = 3f;
                        Main.dust[dust].velocity = dustVelocity * 0.3f;
                    }
                }
            }

            // Charge at the target and spawn holy bombs
            else if (NPC.ai[0] == 2f)
            {
                // Set damage
                NPC.damage = NPC.defDamage;

                if (Math.Sign(NPC.velocity.X) != 0)
                    NPC.spriteDirection = -Math.Sign(NPC.velocity.X);
                NPC.spriteDirection = Math.Sign(NPC.velocity.X);

                NPC.ai[1] += 1f;
                float phaseGateValue = (bossRush || enraged) ? 120f : death ? 140f : revenge ? 150f : expertMode ? 160f : 180f;
                if (NPC.ai[1] >= phaseGateValue)
                {
                    NPC.ai[0] = 3f;

                    // Slown down duration ranges from 30 to 60 frames in rev, otherwise it's always 60 frames
                    float slowDownDurationAfterCharge = revenge ? Main.rand.Next(30, 61) : 60f;
                    NPC.ai[1] = slowDownDurationAfterCharge;
                    NPC.localAI[2] = 0f;
                    NPC.velocity /= 2f;
                    NPC.netUpdate = true;
                }
                else
                {
                    Vector2 targetVector = (player.Center - NPC.Center).SafeNormalize(new Vector2(NPC.direction, 0f));

                    if (NPC.localAI[2] == 0f)
                    {
                        // Accelerate
                        if (NPC.velocity.Length() < maxChargeVelocity)
                        {
                            float velocityMult = (bossRush || enraged) ? 1.04f : death ? 1.036667f : revenge ? 1.035f : expertMode ? 1.033333f : 1.03f;
                            NPC.velocity = targetVector * (NPC.velocity.Length() * velocityMult);
                            if (NPC.velocity.Length() > maxChargeVelocity)
                            {
                                NPC.localAI[2] = 1f;
                                NPC.velocity = NPC.velocity.SafeNormalize(new Vector2(NPC.direction, 0f)) * maxChargeVelocity;
                            }
                        }
                    }
                    else
                    {
                        // Charge towards target
                        float inertia = (bossRush || enraged) ? 57f : death ? 63f : revenge ? 66f : expertMode ? 69f : 75f;
                        NPC.velocity = (NPC.velocity * (inertia - 1f) + targetVector * (NPC.velocity.Length() + (0.111111117f * inertia))) / inertia;
                    }

                    // Lay holy bombs while charging
                    int projectileGateValue = (int)(phaseGateValue * 0.4f);
                    if (NPC.ai[1] % projectileGateValue == 0f)
                    {
                        if (Main.netMode != NetmodeID.MultiplayerClient)
                        {
                            float projectileVelocityY = NPC.velocity.Y;
                            if (projectileVelocityY < 0f)
                                projectileVelocityY = 0f;

                            projectileVelocityY += expertMode ? 4f : 3f;
                            Vector2 projectileVelocity = new Vector2(NPC.velocity.X * 0.25f, projectileVelocityY);
                            int type = ModContent.ProjectileType<HolyBomb>();
                            int damage = NPC.GetProjectileDamage(type);
                            Projectile.NewProjectile(NPC.GetSource_FromAI(), shootFrom, projectileVelocity, type, damage, 0f, Main.myPlayer);
                        }
                    }
                }
            }

            // Slow down and pause for a bit
            else if (NPC.ai[0] == 3f)
            {
                // Avoid cheap bullshit
                NPC.damage = 0;

                if (Math.Sign(NPC.velocity.X) != 0)
                    NPC.spriteDirection = -Math.Sign(NPC.velocity.X);
                NPC.spriteDirection = Math.Sign(NPC.velocity.X);

                NPC.ai[1] -= 1f;
                if (NPC.ai[1] <= 0f)
                {
                    NPC.ai[0] = 1f;
                    NPC.ai[2] += 1f;

                    // Charges either once or twice in a row in rev, otherwise it will always charge twice in a row
                    float totalCharges = revenge ? (Main.rand.Next(2) + 1f) : 2f;
                    bool dontCharge = NPC.ai[2] >= totalCharges;
                    NPC.ai[1] = dontCharge ? -commanderGuardPhase2Duration : 0f;
                    if (dontCharge)
                        NPC.ai[2] = 0f;

                    NPC.TargetClosest();
                    NPC.netUpdate = true;
                }

                NPC.velocity *= 0.97f;
            }

            if (Main.zenithWorld)
            {
                if (Math.Abs(NPC.Center.X - player.Center.X) > 10f)
                {
                    float playerLocation = NPC.Center.X - player.Center.X;
                    NPC.direction = playerLocation < 0f ? 1 : -1;
                    NPC.spriteDirection = NPC.direction;
                }
                // Block the main guardian
                Vector2 guardPos = Main.npc[CalamityGlobalNPC.doughnutBoss].Center;
                Vector2 playerPos = player.Center;
                Vector2 midPoint = ((guardPos - playerPos) / 1.25f) + playerPos;
                NPC.position = midPoint;
                return;
            }
        }

        public override bool CheckActive() => false;

        public override bool PreDraw(SpriteBatch spriteBatch, Vector2 screenPos, Color drawColor)
        {
            void drawGuardianInstance(Vector2 drawOffset, Color? colorOverride)
            {
                SpriteEffects spriteEffects = SpriteEffects.None;
                if (NPC.spriteDirection == 1)
                    spriteEffects = SpriteEffects.FlipHorizontally;

                Texture2D texture2D15 = TextureAssets.Npc[NPC.type].Value;
                Vector2 drawPos = NPC.Center - screenPos;
                Vector2 halfSizeTexture = new Vector2(TextureAssets.Npc[NPC.type].Value.Width / 2, TextureAssets.Npc[NPC.type].Value.Height / Main.npcFrameCount[NPC.type] / 2);
                int afterimageAmt = 5;
                if (NPC.ai[0] == 2f)
                    afterimageAmt = 10;

                if (CalamityClientConfig.Instance.Afterimages)
                {
                    for (int i = 1; i < afterimageAmt; i += 2)
                    {
                        Color afterimageColor = drawColor;
                        afterimageColor = Color.Lerp(afterimageColor, Color.White, 0.5f);
                        afterimageColor = NPC.GetAlpha(afterimageColor);
                        afterimageColor *= (afterimageAmt - i) / 15f;
                        if (colorOverride != null)
                            afterimageColor = colorOverride.Value;

                        Vector2 afterimagePos = NPC.oldPos[i] + new Vector2(NPC.width, NPC.height) / 2f - screenPos;
                        afterimagePos -= new Vector2(texture2D15.Width, texture2D15.Height / Main.npcFrameCount[NPC.type]) * NPC.scale / 2f;
                        afterimagePos += halfSizeTexture * NPC.scale + new Vector2(0f, NPC.gfxOffY) + drawOffset;
                        spriteBatch.Draw(texture2D15, afterimagePos, NPC.frame, afterimageColor, NPC.rotation, halfSizeTexture, NPC.scale, spriteEffects, 0f);
                    }
                }

                Vector2 drawLocation = drawPos;
                drawLocation -= new Vector2(texture2D15.Width, texture2D15.Height / Main.npcFrameCount[NPC.type]) * NPC.scale / 2f;
                drawLocation += halfSizeTexture * NPC.scale + new Vector2(0f, NPC.gfxOffY) + drawOffset;
                spriteBatch.Draw(texture2D15, drawLocation, NPC.frame, colorOverride ?? NPC.GetAlpha(drawColor), NPC.rotation, halfSizeTexture, NPC.scale, spriteEffects, 0f);

                texture2D15 = Texture_Glow.Value;
                Color timeBasedGlowColor = Color.Lerp(Color.White, Color.Yellow, 0.5f);
                if (NPC.Calamity().CurrentlyEnraged)
                {
                    texture2D15 = TextureNight_Glow.Value;
                    timeBasedGlowColor = Color.Lerp(Color.White, Color.Cyan, 0.75f);
                }
                if (Main.zenithWorld)
                {
                    texture2D15 = TextureNight_Glow.Value;
                    timeBasedGlowColor = Main.DiscoColor;
                }
                if (colorOverride != null)
                    timeBasedGlowColor = colorOverride.Value;

                if (CalamityClientConfig.Instance.Afterimages)
                {
                    for (int j = 1; j < afterimageAmt; j++)
                    {
                        Color timeBasedAfterimageColor = timeBasedGlowColor;
                        timeBasedAfterimageColor = Color.Lerp(timeBasedAfterimageColor, Color.White, 0.5f);
                        timeBasedAfterimageColor = NPC.GetAlpha(timeBasedAfterimageColor);
                        timeBasedAfterimageColor *= (afterimageAmt - j) / 15f;
                        if (colorOverride != null)
                            timeBasedAfterimageColor = colorOverride.Value;

                        Vector2 timeBasedAfterimagePos = NPC.oldPos[j] + new Vector2(NPC.width, NPC.height) / 2f - screenPos;
                        timeBasedAfterimagePos -= new Vector2(texture2D15.Width, texture2D15.Height / Main.npcFrameCount[NPC.type]) * NPC.scale / 2f;
                        timeBasedAfterimagePos += halfSizeTexture * NPC.scale + new Vector2(0f, NPC.gfxOffY) + drawOffset;
                        spriteBatch.Draw(texture2D15, timeBasedAfterimagePos, NPC.frame, timeBasedAfterimageColor, NPC.rotation, halfSizeTexture, NPC.scale, spriteEffects, 0f);
                    }
                }

                spriteBatch.Draw(texture2D15, drawLocation, NPC.frame, timeBasedGlowColor, NPC.rotation, halfSizeTexture, NPC.scale, spriteEffects, 0f);
            }

            // Draw laser effects
            float useLaserGateValue = 120f;
            float stopLaserGateValue = (CalamityWorld.revenge || BossRushEvent.BossRushActive) ? 235f : 315f;
            float maxIntensity = 45f;
            float increaseIntensityGateValue = useLaserGateValue - maxIntensity;
            float decreaseIntensityGateValue = stopLaserGateValue - maxIntensity;
            if (!NPC.IsABestiaryIconDummy)
            {
                bool usingLaser = Main.npc[CalamityGlobalNPC.doughnutBoss].ai[0] == 5f;
                if (usingLaser)
                {
                    bool increaseIntensity = Main.npc[CalamityGlobalNPC.doughnutBoss].ai[1] > increaseIntensityGateValue;
                    bool decreaseIntensity = Main.npc[CalamityGlobalNPC.doughnutBoss].ai[1] > decreaseIntensityGateValue;
                    float burnIntensity = decreaseIntensity ? Utils.GetLerpValue(0f, maxIntensity, maxIntensity - (Main.npc[CalamityGlobalNPC.doughnutBoss].ai[1] - decreaseIntensityGateValue), true) : Utils.GetLerpValue(0f, maxIntensity, Main.npc[CalamityGlobalNPC.doughnutBoss].ai[1], true);
                    int totalGuardiansToDraw = (int)MathHelper.Lerp(1f, 30f, burnIntensity);
                    for (int i = 0; i < totalGuardiansToDraw; i++)
                    {
                        float offsetAngle = MathHelper.TwoPi * i * 2f / totalGuardiansToDraw;
                        float drawOffsetFactor = (float)Math.Sin(offsetAngle * 6f + Main.GlobalTimeWrappedHourly * MathHelper.Pi);
                        drawOffsetFactor *= (float)Math.Pow(burnIntensity, 3f) * 50f;

                        Vector2 drawOffset = offsetAngle.ToRotationVector2() * drawOffsetFactor;
                        Color baseColor = Color.White * (MathHelper.Lerp(0.4f, 0.8f, burnIntensity) / totalGuardiansToDraw * 1.5f);
                        baseColor.A = 0;

                        baseColor = Color.Lerp(Color.White, baseColor, burnIntensity);
                        drawGuardianInstance(drawOffset, totalGuardiansToDraw == 1 ? null : baseColor);
                    }
                }
                else
                    drawGuardianInstance(Vector2.Zero, null);
            }
            else
                drawGuardianInstance(Vector2.Zero, null);

            if (NPC.IsABestiaryIconDummy)
                return false;

            // Draw shields while healer is alive
            if (NPC.localAI[1] < TimeForShieldDespawn)
            {
                float maxOscillation = 60f;
                float minScale = 0.8f;
                float maxPulseScale = 1f - minScale;
                float minOpacity = 0.5f;
                float maxOpacityScale = 1f - minOpacity;
                float currentOscillation = MathHelper.Lerp(0f, maxOscillation, ((float)Math.Sin(Main.GlobalTimeWrappedHourly * MathHelper.Pi) + 1f) * 0.5f);
                float shieldOpacity = minOpacity + maxOpacityScale * Utils.Remap(currentOscillation, 0f, maxOscillation, 1f, 0f);
                float oscillationRatio = currentOscillation / maxOscillation;
                float invertedOscillationRatio = 1f - (1f - oscillationRatio) * (1f - oscillationRatio);
                float oscillationScale = 1f - (1f - invertedOscillationRatio) * (1f - invertedOscillationRatio);
                float remappedOscillation = Utils.Remap(currentOscillation, maxOscillation - 15f, maxOscillation, 0f, 1f);
                float twoOscillationsMultipliedTogetherForScaleCalculation = remappedOscillation * remappedOscillation;
                float invertedOscillationUsedForScale = MathHelper.Lerp(minScale, 1f, 1f - twoOscillationsMultipliedTogetherForScaleCalculation);
                float shieldScale = (minScale + maxPulseScale * oscillationScale) * invertedOscillationUsedForScale;
                float smallerRemappedOscillation = Utils.Remap(currentOscillation, 20f, maxOscillation, 0f, 1f);
                float invertedSmallerOscillationRatio = 1f - (1f - smallerRemappedOscillation) * (1f - smallerRemappedOscillation);
                float smallerOscillationScale = 1f - (1f - invertedSmallerOscillationRatio) * (1f - invertedSmallerOscillationRatio);
                float shieldScale2 = (minScale + maxPulseScale * smallerOscillationScale) * invertedOscillationUsedForScale;
                Texture2D shieldTexture = ModContent.Request<Texture2D>("CalamityMod/ExtraTextures/GreyscaleOpenCircle").Value;
                Rectangle shieldFrame = shieldTexture.Frame();
                Vector2 origin = shieldFrame.Size() * 0.5f;
                Vector2 shieldDrawPos = NPC.Center - screenPos;
                shieldDrawPos -= new Vector2(shieldTexture.Width, shieldTexture.Height) * NPC.scale / 2f;
                shieldDrawPos += origin * NPC.scale + new Vector2(0f, NPC.gfxOffY);
                float minHue = NPC.Calamity().CurrentlyEnraged ? 0.4f : 0.06f;
                float maxHue = minHue + 0.12f;
                float opacityScaleDuringShieldDespawn = (TimeForShieldDespawn - NPC.localAI[1]) / TimeForShieldDespawn;
                float scaleDuringShieldDespawnScale = 1.8f;
                float scaleDuringShieldDespawn = (1f - opacityScaleDuringShieldDespawn) * scaleDuringShieldDespawnScale;
                float colorScale = MathHelper.Lerp(0f, shieldOpacity, opacityScaleDuringShieldDespawn);
                Color color = Main.hslToRgb(MathHelper.Lerp(NPC.Calamity().CurrentlyEnraged ? minHue : maxHue - minHue, maxHue, ((float)Math.Sin(Main.GlobalTimeWrappedHourly * MathHelper.TwoPi) + 1f) * 0.5f), 1f, 0.5f) * colorScale;
                Color color2 = Main.hslToRgb(MathHelper.Lerp(minHue, NPC.Calamity().CurrentlyEnraged ? maxHue : maxHue - minHue, ((float)Math.Sin(Main.GlobalTimeWrappedHourly * MathHelper.Pi * 3f) + 1f) * 0.5f), 1f, 0.5f) * colorScale;
                color2.A = 0;
                color *= 0.6f;
                color2 *= 0.6f;
                float scaleMult = 1.2f + scaleDuringShieldDespawn;

                // The scale used for the noise overlay polygons also grows and shrinks
                // This is intentionally out of sync with the shield, and intentionally desynced per player
                // Don't put this anywhere less than 0.25f or higher than 1f. The higher it is, the denser / more zoomed out the noise overlay is.
                float noiseScale = MathHelper.Lerp(0.4f, 0.8f, (float)Math.Sin(Main.GlobalTimeWrappedHourly * 0.3f) * 0.5f + 0.5f);

                // Define shader parameters
                Effect shieldEffect = Filters.Scene["CalamityMod:RoverDriveShield"].GetShader().Shader;
                shieldEffect.Parameters["time"].SetValue(Main.GlobalTimeWrappedHourly * 0.058f); // Scrolling speed of polygonal overlay
                shieldEffect.Parameters["blowUpPower"].SetValue(2.8f);
                shieldEffect.Parameters["blowUpSize"].SetValue(0.4f);
                shieldEffect.Parameters["noiseScale"].SetValue(noiseScale);

                shieldEffect.Parameters["shieldOpacity"].SetValue(opacityScaleDuringShieldDespawn);
                shieldEffect.Parameters["shieldEdgeBlendStrenght"].SetValue(4f);

                Color edgeColor = CalamityUtils.MulticolorLerp(Main.GlobalTimeWrappedHourly * 0.2f, color, color2);

                // Define shader parameters for shield color
                shieldEffect.Parameters["shieldColor"].SetValue(color.ToVector3());
                shieldEffect.Parameters["shieldEdgeColor"].SetValue(edgeColor.ToVector3());

                spriteBatch.Draw(shieldTexture, shieldDrawPos, shieldFrame, color, NPC.rotation, origin, shieldScale2 * scaleMult, SpriteEffects.None, 0f);
                spriteBatch.Draw(shieldTexture, shieldDrawPos, shieldFrame, color2, NPC.rotation, origin, shieldScale2 * scaleMult * 0.95f, SpriteEffects.None, 0f);
                spriteBatch.Draw(shieldTexture, shieldDrawPos, shieldFrame, color, NPC.rotation, origin, shieldScale * scaleMult, SpriteEffects.None, 0f);
                spriteBatch.Draw(shieldTexture, shieldDrawPos, shieldFrame, color2, NPC.rotation, origin, shieldScale * scaleMult * 0.95f, SpriteEffects.None, 0f);

                // The shield for the border MUST be drawn before the main shield, it becomes incredibly visually obnoxious otherwise.

                Main.spriteBatch.End();
                Main.spriteBatch.Begin(SpriteSortMode.Immediate, BlendState.Additive, Main.DefaultSamplerState, DepthStencilState.None, Main.Rasterizer, shieldEffect, Main.GameViewMatrix.TransformationMatrix);
                // Fetch shield heat overlay texture (this is the neutrons fed to the shader)
                Texture2D heatTex = ModContent.Request<Texture2D>("CalamityMod/ExtraTextures/GreyscaleGradients/Neurons2").Value;
                Vector2 pos = NPC.Center + NPC.gfxOffY * Vector2.UnitY - Main.screenPosition;
                Main.spriteBatch.Draw(heatTex, shieldDrawPos, null, Color.White, 0, heatTex.Size() / 2f, shieldScale * scaleMult * 0.5f, 0, 0);
            }

            return false;
        }

        public override void ModifyNPCLoot(NPCLoot npcLoot) => npcLoot.Add(ModContent.ItemType<RelicOfResilience>(), 4);

        // Can only hit the target if within certain distance
        public override bool CanHitPlayer(Player target, ref int cooldownSlot)
        {
            cooldownSlot = ImmunityCooldownID.Bosses;

            Rectangle targetHitbox = target.Hitbox;

            float hitboxTopLeft = Vector2.Distance(NPC.Center, targetHitbox.TopLeft());
            float hitboxTopRight = Vector2.Distance(NPC.Center, targetHitbox.TopRight());
            float hitboxBotLeft = Vector2.Distance(NPC.Center, targetHitbox.BottomLeft());
            float hitboxBotRight = Vector2.Distance(NPC.Center, targetHitbox.BottomRight());

            float minDist = hitboxTopLeft;
            if (hitboxTopRight < minDist)
                minDist = hitboxTopRight;
            if (hitboxBotLeft < minDist)
                minDist = hitboxBotLeft;
            if (hitboxBotRight < minDist)
                minDist = hitboxBotRight;

            return minDist <= 80f;
        }

        public override void OnHitPlayer(Player target, Player.HurtInfo hurtInfo)
        {
            if (hurtInfo.Damage > 0)
                target.AddBuff(ModContent.BuffType<HolyFlames>(), 160, true);
        }

        public override void ModifyHitByProjectile(Projectile projectile, ref NPC.HitModifiers modifiers)
        {
            // eat projectiles but take more damage based on piercing in the zenith seed
            if (Main.zenithWorld && !projectile.minion)
            {
                if (projectile.penetrate <= -1 || projectile.penetrate > 5)
                {
                    modifiers.SourceDamage *= 2.5f;
                }
                else
                {
                    modifiers.SourceDamage *= projectile.penetrate / 2f;
                }
                projectile.active = false;
            }
        }

        public override void HitEffect(NPC.HitInfo hit)
        {
            for (int k = 0; k < 5; k++)
                Dust.NewDust(NPC.position, NPC.width, NPC.height, (int)CalamityDusts.ProfanedFire, hit.HitDirection, -1f, 0, default, 1f);

            if (NPC.life <= 0)
            {
                if (Main.netMode != NetmodeID.Server)
                {
                    Gore.NewGore(NPC.GetSource_Death(), NPC.position, NPC.velocity, Mod.Find<ModGore>("ProfanedGuardianBossT").Type, 1f);
                    Gore.NewGore(NPC.GetSource_Death(), NPC.position, NPC.velocity, Mod.Find<ModGore>("ProfanedGuardianBossT2").Type, 1f);
                    Gore.NewGore(NPC.GetSource_Death(), NPC.position, NPC.velocity, Mod.Find<ModGore>("ProfanedGuardianBossT3").Type, 1f);
                    Gore.NewGore(NPC.GetSource_Death(), NPC.position, NPC.velocity, Mod.Find<ModGore>("ProfanedGuardianBossT4").Type, 1f);
                }

                for (int k = 0; k < 50; k++)
                    Dust.NewDust(NPC.position, NPC.width, NPC.height, (int)CalamityDusts.ProfanedFire, hit.HitDirection, -1f, 0, default, 1f);
            }
        }
    }
}
